/*
 * TurtleGraphicsUtils.java
 *
 * Created on February 22, 2003, 4:29 PM
 *
 * Copyright 2003 Danny Brewer
 * Anyone may run this code.
 * If you wish to modify or distribute this code, then
 *  you are granted a license to do so only under the terms
 *  of the Gnu Lesser General Public License.
 * See:  http://www.gnu.org/licenses/lgpl.html
 */

package nom.DannyBrewer.graphics;

/**
 *
 * @author  danny brewer
 */
public class TurtleGraphicsUtils {
    private TurtleGraphicsUtils() { }
    
    
    /*
     * Draw a polygon by doing line(), turn(), line(), turn(), etc..
     * For instance, to draw a square with sides of length 5,
     *  you would do...
     *      lineForward( 5 );    turn( 90 );  // 90 degree turn
     *      lineForward( 5 );    turn( 90 );  // 90 degree turn
     *      lineForward( 5 );    turn( 90 );  // 90 degree turn
     *      lineForward( 5 );    turn( 90 );  // 90 degree turn
     * The turtle is left at its original position,
     *  pointed in the original direction.
     * The turtle might not end up on the EXACT same pixel as it started,
     *  this depends on the accuracy of the line() and turn() operations.
     */
    public static void drawPolygon( TurtleGraphics turtle,
    int numSides, int sideLength ) {
        double fullCircle = turtle.getFullCircleAngle();
        double turnAngle = fullCircle / numSides;
        for( int i = 0;  i < numSides;  ++i ) {
            turtle.lineForward( sideLength );
            turtle.turn( turnAngle );
        }
    }
    
    
    /*
     * Draw a star inside of the polygon that would be drawn if you had
     *  called drawPolygon with the same parameters.
     * The star is drawn by drawing a line between points of the polyggon.
     * A line is drawn from one point to another point which is two points
     *  further along the polygon
     */
    public static void inscribeStar( TurtleGraphics turtle,
    int numSides, int sideLength ) {
        inscribeStar( turtle, numSides, sideLength, 2 );
    }
    /*
     * Similar to inscribeStar, but instead of skipping two points ahead
     *  along the polygon, allows you to specify how many points ahead to skip.
     * A line is drawn from one point to another point which is nSkipPoints
     *  further along the polygon.  For example, if you pass 3 for nSkipPoints,
     *  then a line will be drawn that connects every third vertex along the
     *  ploygon's perimeter -- thus inscribing a "star" inside the polygon.
     * If you pass 1 for skipPoints, then this routine behaves EXACTLY like
     *  drawPolygon().
     */
    public static void inscribeStar( TurtleGraphics turtle,
    int numSides, int sideLength,
    int skipPoints ) {
        int[] x = new int[ numSides ];
        int[] y = new int[ numSides ];
        
        // Get the x,y coordinates of each point that makes up the enclosing polygon.
        collectPolygonPoints( turtle, numSides, sideLength, x, y );
        
        // None of those points have been visited so far.
        boolean[] visited = new boolean[ numSides ];
        int pointsVisited = 0;
        
        // Start on the first point.
        int point = 0;
        
        while( pointsVisited < numSides ) {
            // Find the next un-visited point.
            while( visited[ point ] ) point = point + 1;
            
            // Start at this point
            turtle.jumpTo( x[point], y[point] );
            
            // Inscribe a star.
            while( ! visited[ point ] ) {
                // This point is now visited.
                visited[ point ] = true;
                pointsVisited = pointsVisited + 1;
                
                // Move to next point.
                point = point + skipPoints;
                point = point % numSides;
                
                turtle.lineTo( x[point], y[point] );
            }
        }
        
        // Return to original location.
        turtle.jumpTo( x[0], y[0] );
    }
    
    
    
    /*
     *	Calculate the radius for a polygon.
     *	The polygon is not actually drawn.
     *	The turtle is left in its original position and orientation.
     */
    public static int calcPolygonRadius( TurtleGraphics turtle,
    int numSides, int sideLength ) {
        double fullCircle = turtle.getFullCircleAngle();
        double turnAngle = fullCircle / numSides;
        
        // Average all of the polygon vertexes to calculate the center point.
        int centerX = 0, centerY = 0;
        for( int i = 0;  i < numSides;  ++i ) {
            centerX = centerX + turtle.getTurtleXPosition();
            centerY = centerY + turtle.getTurtleYPosition();
            turtle.jumpForward( sideLength );
            turtle.turn( turnAngle );
        }
        centerX = centerX / numSides;
        centerY = centerY / numSides;
        
        // Calculate the initial polygon point's X and Y distance from the center point.
        int dXfromCenter = Math.abs( turtle.getTurtleXPosition() - centerX );
        int dYfromCenter = Math.abs( turtle.getTurtleYPosition() - centerY );
        
        // Use pythagorean therom to find distance between two points.
        // Distance between (dX,dY) and (centerX,centerY).
        int radius = (int) Math.sqrt( dXfromCenter * dXfromCenter + dYfromCenter * dYfromCenter );
        
        return radius;
    }
    
    
    /*
     * Collect the points of a polygon.
     * Works like drawPolygon, but no polygon is drawn.
     * The lineForward() operation is replaced by jumpForward().
     * Each point visited is gathered into the two arrays x[] and y[],
     *  both of which must be dimensioned to hold numSides elements.
     */
    public static void collectPolygonPoints( TurtleGraphics turtle,
    int numSides, int sideLength,
    int[] x, int[] y ) {
        double fullCircle = turtle.getFullCircleAngle();
        double turnAngle = fullCircle / numSides;
        for( int i = 0;  i < numSides;  ++i ) {
            x[i] = turtle.getTurtleXPosition();
            y[i] = turtle.getTurtleYPosition();
            turtle.jumpForward( sideLength );
            turtle.turn( turnAngle );
        }
    }
    
    
    
}
